// SPDX-License-Identifier: GPL-2.0-only
/*
 *  Common functions for kernel modules using Dell SMBIOS
 *
 *  Copyright (c) Red Hat <mjg@redhat.com>
 *  Copyright (c) 2014 Gabriele Mazzotta <gabriele.mzt@gmail.com>
 *  Copyright (c) 2014 Pali Rohár <pali@kernel.org>
 *
 *  Based on documentation in the libsmbios package:
 *  Copyright (C) 2005-2014 Dell Inc.
 */
#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/container_of.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/capability.h>
#include <linux/dmi.h>
#include <linux/err.h>
#include <linux/mutex.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include "dell-smbios.h"

static u32 da_supported_commands;
static int da_num_tokens;
static struct platform_device *platform_device;
static struct calling_interface_token *da_tokens;
static struct token_sysfs_data *token_entries;
static struct attribute **token_attrs;
static DEFINE_MUTEX(smbios_mutex);

struct token_sysfs_data {
	struct device_attribute location_attr;
	struct device_attribute value_attr;
	struct calling_interface_token *token;
};

struct smbios_device {
	struct list_head list;
	struct device *device;
	int priority;
	int (*call_fn)(struct calling_interface_buffer *arg);
};

struct smbios_call {
	u32 need_capability;
	int cmd_class;
	int cmd_select;
};

/* calls that are whitelisted for given capabilities */
static struct smbios_call call_whitelist[] = {
	/* generally tokens are allowed, but may be further filtered or
	 * restricted by token blacklist or whitelist
	 */
	{CAP_SYS_ADMIN,	CLASS_TOKEN_READ,	SELECT_TOKEN_STD},
	{CAP_SYS_ADMIN,	CLASS_TOKEN_READ,	SELECT_TOKEN_AC},
	{CAP_SYS_ADMIN,	CLASS_TOKEN_READ,	SELECT_TOKEN_BAT},
	{CAP_SYS_ADMIN,	CLASS_TOKEN_WRITE,	SELECT_TOKEN_STD},
	{CAP_SYS_ADMIN,	CLASS_TOKEN_WRITE,	SELECT_TOKEN_AC},
	{CAP_SYS_ADMIN,	CLASS_TOKEN_WRITE,	SELECT_TOKEN_BAT},
	/* used by userspace: fwupdate */
	{CAP_SYS_ADMIN, CLASS_ADMIN_PROP,	SELECT_ADMIN_PROP},
	/* used by userspace: fwupd */
	{CAP_SYS_ADMIN,	CLASS_INFO,		SELECT_DOCK},
	{CAP_SYS_ADMIN,	CLASS_FLASH_INTERFACE,	SELECT_FLASH_INTERFACE},
};

/* calls that are explicitly blacklisted */
static struct smbios_call call_blacklist[] = {
	{0x0000,  1,  7}, /* manufacturing use */
	{0x0000,  6,  5}, /* manufacturing use */
	{0x0000, 11,  3}, /* write once */
	{0x0000, 11,  7}, /* write once */
	{0x0000, 11, 11}, /* write once */
	{0x0000, 19, -1}, /* diagnostics */
	/* handled by kernel: dell-laptop */
	{0x0000, CLASS_INFO, SELECT_RFKILL},
	{0x0000, CLASS_KBD_BACKLIGHT, SELECT_KBD_BACKLIGHT},
	{0x0000, CLASS_INFO, SELECT_THERMAL_MANAGEMENT},
};

struct token_range {
	u32 need_capability;
	u16 min;
	u16 max;
};

/* tokens that are whitelisted for given capabilities */
static struct token_range token_whitelist[] = {
	/* used by userspace: fwupdate */
	{CAP_SYS_ADMIN,	CAPSULE_EN_TOKEN,	CAPSULE_DIS_TOKEN},
	/* can indicate to userspace that WMI is needed */
	{0x0000,	WSMT_EN_TOKEN,		WSMT_DIS_TOKEN}
};

/* tokens that are explicitly blacklisted */
static struct token_range token_blacklist[] = {
	{0x0000, 0x0058, 0x0059}, /* ME use */
	{0x0000, 0x00CD, 0x00D0}, /* raid shadow copy */
	{0x0000, 0x013A, 0x01FF}, /* sata shadow copy */
	{0x0000, 0x0175, 0x0176}, /* write once */
	{0x0000, 0x0195, 0x0197}, /* diagnostics */
	{0x0000, 0x01DC, 0x01DD}, /* manufacturing use */
	{0x0000, 0x027D, 0x0284}, /* diagnostics */
	{0x0000, 0x02E3, 0x02E3}, /* manufacturing use */
	{0x0000, 0x02FF, 0x02FF}, /* manufacturing use */
	{0x0000, 0x0300, 0x0302}, /* manufacturing use */
	{0x0000, 0x0325, 0x0326}, /* manufacturing use */
	{0x0000, 0x0332, 0x0335}, /* fan control */
	{0x0000, 0x0350, 0x0350}, /* manufacturing use */
	{0x0000, 0x0363, 0x0363}, /* manufacturing use */
	{0x0000, 0x0368, 0x0368}, /* manufacturing use */
	{0x0000, 0x03F6, 0x03F7}, /* manufacturing use */
	{0x0000, 0x049E, 0x049F}, /* manufacturing use */
	{0x0000, 0x04A0, 0x04A3}, /* disagnostics */
	{0x0000, 0x04E6, 0x04E7}, /* manufacturing use */
	{0x0000, 0x4000, 0x7FFF}, /* internal BIOS use */
	{0x0000, 0x9000, 0x9001}, /* internal BIOS use */
	{0x0000, 0xA000, 0xBFFF}, /* write only */
	{0x0000, 0xEFF0, 0xEFFF}, /* internal BIOS use */
	/* handled by kernel: dell-laptop */
	{0x0000, BRIGHTNESS_TOKEN,	BRIGHTNESS_TOKEN},
	{0x0000, KBD_LED_OFF_TOKEN,	KBD_LED_AUTO_TOKEN},
	{0x0000, KBD_LED_AC_TOKEN,	KBD_LED_AC_TOKEN},
	{0x0000, KBD_LED_AUTO_25_TOKEN,	KBD_LED_AUTO_75_TOKEN},
	{0x0000, KBD_LED_AUTO_100_TOKEN,	KBD_LED_AUTO_100_TOKEN},
	{0x0000, GLOBAL_MIC_MUTE_ENABLE,	GLOBAL_MIC_MUTE_DISABLE},
};

static LIST_HEAD(smbios_device_list);

int dell_smbios_error(int value)
{
	switch (value) {
	case 0: /* Completed successfully */
		return 0;
	case -1: /* Completed with error */
		return -EIO;
	case -2: /* Function not supported */
		return -ENXIO;
	default: /* Unknown error */
		return -EINVAL;
	}
}
EXPORT_SYMBOL_GPL(dell_smbios_error);

int dell_smbios_register_device(struct device *d, int priority, void *call_fn)
{
	struct smbios_device *priv;

	priv = devm_kzalloc(d, sizeof(struct smbios_device), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;
	get_device(d);
	priv->device = d;
	priv->priority = priority;
	priv->call_fn = call_fn;
	mutex_lock(&smbios_mutex);
	list_add_tail(&priv->list, &smbios_device_list);
	mutex_unlock(&smbios_mutex);
	dev_dbg(d, "Added device: %s\n", d->driver->name);
	return 0;
}
EXPORT_SYMBOL_GPL(dell_smbios_register_device);

void dell_smbios_unregister_device(struct device *d)
{
	struct smbios_device *priv;

	mutex_lock(&smbios_mutex);
	list_for_each_entry(priv, &smbios_device_list, list) {
		if (priv->device == d) {
			list_del(&priv->list);
			put_device(d);
			break;
		}
	}
	mutex_unlock(&smbios_mutex);
	dev_dbg(d, "Remove device: %s\n", d->driver->name);
}
EXPORT_SYMBOL_GPL(dell_smbios_unregister_device);

int dell_smbios_call_filter(struct device *d,
			    struct calling_interface_buffer *buffer)
{
	u16 t = 0;
	int i;

	/* can't make calls over 30 */
	if (buffer->cmd_class > 30) {
		dev_dbg(d, "class too big: %u\n", buffer->cmd_class);
		return -EINVAL;
	}

	/* supported calls on the particular system */
	if (!(da_supported_commands & (1 << buffer->cmd_class))) {
		dev_dbg(d, "invalid command, supported commands: 0x%8x\n",
			da_supported_commands);
		return -EINVAL;
	}

	/* match against call blacklist  */
	for (i = 0; i < ARRAY_SIZE(call_blacklist); i++) {
		if (buffer->cmd_class != call_blacklist[i].cmd_class)
			continue;
		if (buffer->cmd_select != call_blacklist[i].cmd_select &&
		    call_blacklist[i].cmd_select != -1)
			continue;
		dev_dbg(d, "blacklisted command: %u/%u\n",
			buffer->cmd_class, buffer->cmd_select);
		return 